<?php

require_once __DIR__ . '/../db_connect.php';

class UserService {
    private $conn;

    public function __construct($db_connection) {
        $this->conn = $db_connection;
    }

    public function loginUser($username, $password) {
        $stmt = $this->conn->prepare("SELECT id, username, password FROM users WHERE username = ?");
        $stmt->bind_param("s", $username);
        $stmt->execute();
        $result = $stmt->get_result();

        if ($result->num_rows === 1) {
            $user = $result->fetch_assoc();
            // Şifreyi doğrula
            if (password_verify($password, $user['password'])) {
                $stmt->close();
                // Oturum başlatma veya token oluşturma burada yapılabilir
                return ['status' => 'success', 'message' => "Giriş başarılı!", 'user_id' => $user['id'], 'username' => $user['username']];
            } else {
                $stmt->close();
                return ['status' => 'error', 'message' => "Yanlış şifre."];
            }
        } else {
            $stmt->close();
            return ['status' => 'error', 'message' => "Kullanıcı bulunamadı."];
        }
    }

    public function registerUser($username, $password) {
        // Kullanıcı adının benzersizliğini kontrol et
        if ($this->isUsernameTaken($username)) {
            return ['status' => 'error', 'message' => "Bu kullanıcı adı zaten alınmış."];
        }

        // Şifreyi hash'le
        $hashed_password = password_hash($password, PASSWORD_DEFAULT);

        $stmt = $this->conn->prepare("INSERT INTO users (username, password) VALUES (?, ?)");
        $stmt->bind_param("ss", $username, $hashed_password);

        if ($stmt->execute()) {
            $newId = $stmt->insert_id;
            $stmt->close();
            return ['status' => 'success', 'message' => "Kullanıcı başarıyla kaydedildi!", 'id' => $newId];
        } else {
            $error = $stmt->error;
            $stmt->close();
            return ['status' => 'error', 'message' => "Kullanıcı kaydedilirken hata: " . $error];
        }
    }

    public function updateUsername($userId, $newUsername) {
        // Yeni kullanıcı adının benzersizliğini kontrol et (kendisi hariç)
        if ($this->isUsernameTaken($newUsername, $userId)) {
            return ['status' => 'error', 'message' => "Bu kullanıcı adı zaten başka bir kullanıcı tarafından kullanılıyor."];
        }

        $stmt = $this->conn->prepare("UPDATE users SET username = ? WHERE id = ?");
        $stmt->bind_param("si", $newUsername, $userId);

        if ($stmt->execute()) {
            $affected_rows = $stmt->affected_rows;
            $stmt->close();
            if ($affected_rows > 0) {
                return ['status' => 'success', 'message' => "Kullanıcı adı başarıyla güncellendi!"];
            } else {
                return ['status' => 'info', 'message' => "Güncellenecek kullanıcı bulunamadı veya kullanıcı adı değişmedi."];
            }
        } else {
            $error = $stmt->error;
            $stmt->close();
            return ['status' => 'error', 'message' => "Kullanıcı adı güncellenirken hata: " . $error];
        }
    }

    public function updatePassword($userId, $newPassword) {
        $hashed_password = password_hash($newPassword, PASSWORD_DEFAULT);

        $stmt = $this->conn->prepare("UPDATE users SET password = ? WHERE id = ?");
        $stmt->bind_param("si", $hashed_password, $userId);

        if ($stmt->execute()) {
            $affected_rows = $stmt->affected_rows;
            $stmt->close();
            if ($affected_rows > 0) {
                return ['status' => 'success', 'message' => "Şifre başarıyla güncellendi!"];
            } else {
                return ['status' => 'info', 'message' => "Güncellenecek kullanıcı bulunamadı veya şifre değişmedi."];
            }
        } else {
            $error = $stmt->error;
            $stmt->close();
            return ['status' => 'error', 'message' => "Şifre güncellenirken hata: " . $error];
        }
    }

    public function deleteUser($userId) {
        $stmt = $this->conn->prepare("DELETE FROM users WHERE id = ?");
        $stmt->bind_param("i", $userId);

        if ($stmt->execute()) {
            $affected_rows = $stmt->affected_rows;
            $stmt->close();
            if ($affected_rows > 0) {
                return ['status' => 'success', 'message' => "Kullanıcı başarıyla silindi!"];
            } else {
                return ['status' => 'error', 'message' => "Silinecek kullanıcı bulunamadı."];
            }
        } else {
            $error = $stmt->error;
            $stmt->close();
            return ['status' => 'error', 'message' => "Kullanıcı silinirken hata: " . $error];
        }
    }

    private function isUsernameTaken($username, $excludeId = null) {
        $sql = "SELECT id FROM users WHERE username = ?";
        $params = [$username];
        $types = "s";

        if ($excludeId !== null) {
            $sql .= " AND id != ?";
            $params[] = $excludeId;
            $types .= "i";
        }

        $stmt = $this->conn->prepare($sql);
        $stmt->bind_param($types, ...$params);
        $stmt->execute();
        $stmt->store_result();
        $is_taken = $stmt->num_rows > 0;
        $stmt->close();
        return $is_taken;
    }

}

?> 