<?php

require_once __DIR__ . '/../db_connect.php';

class SubcategoryService {
    private $conn;

    public function __construct($db_connection) {
        $this->conn = $db_connection;
    }

    public function addSubcategory($categoryId, $name, $description) {
        // Alt kategori adının benzersizliğini kontrol et (aynı kategori içinde)
        if ($this->isSubcategoryNameTaken($categoryId, $name)) {
            return ['status' => 'error', 'message' => "Bu alt kategori adı seçilen kategori içinde zaten mevcut."];
        }

        $stmt = $this->conn->prepare("INSERT INTO subcategories (categoryId, name, description) VALUES (?, ?, ?)");
        $stmt->bind_param("iss", $categoryId, $name, $description);

        if ($stmt->execute()) {
            $newId = $stmt->insert_id;
            $stmt->close();
            return ['status' => 'success', 'message' => "Alt kategori başarıyla eklendi!", 'id' => $newId];
        } else {
            $error = $stmt->error;
            $stmt->close();
            return ['status' => 'error', 'message' => "Alt kategori eklenirken hata: " . $error];
        }
    }

    public function updateSubcategory($id, $categoryId, $name, $description) {
        // Alt kategori adının benzersizliğini kontrol et (aynı kategori içinde, kendisi hariç)
        if ($this->isSubcategoryNameTaken($categoryId, $name, $id)) {
            return ['status' => 'error', 'message' => "Bu alt kategori adı seçilen kategori içinde zaten başka bir alt kategori tarafından kullanılıyor."];
        }

        $stmt = $this->conn->prepare("UPDATE subcategories SET categoryId = ?, name = ?, description = ? WHERE id = ?");
        $stmt->bind_param("issi", $categoryId, $name, $description, $id);

        if ($stmt->execute()) {
            $affected_rows = $stmt->affected_rows;
            $stmt->close();
            if ($affected_rows > 0) {
                return ['status' => 'success', 'message' => "Alt kategori başarıyla güncellendi!"];
            } else {
                return ['status' => 'info', 'message' => "Güncellenecek alt kategori bulunamadı veya veri değişmedi."];
            }
        } else {
            $error = $stmt->error;
            $stmt->close();
            return ['status' => 'error', 'message' => "Alt kategori güncellenirken hata: " . $error];
        }
    }

    public function deleteSubcategory($id) {
        // Bu alt kategoriye bağlı soru paketleri veya soru-cevaplar olup olmadığını kontrol et (isteğe bağlı, iş mantığına göre)
        // Şu an için sadece alt kategoriyi silme işlemi yapıyorum, bağlı kayıtlar CASCADE ile silinmeli.
        $stmt = $this->conn->prepare("DELETE FROM subcategories WHERE id = ?");
        $stmt->bind_param("i", $id);

        if ($stmt->execute()) {
            $affected_rows = $stmt->affected_rows;
            $stmt->close();
            if ($affected_rows > 0) {
                return ['status' => 'success', 'message' => "Alt kategori başarıyla silindi!"];
            } else {
                return ['status' => 'error', 'message' => "Silinecek alt kategori bulunamadı."];
            }
        } else {
            $error = $stmt->error;
            $stmt->close();
            return ['status' => 'error', 'message' => "Alt kategori silinirken hata: " . $error];
        }
    }

    public function getSubcategoryById($id) {
        $stmt = $this->conn->prepare("SELECT id, categoryId, name, description FROM subcategories WHERE id = ?");
        $stmt->bind_param("i", $id);
        $stmt->execute();
        $result = $stmt->get_result();
        $data = $result->fetch_assoc();
        $stmt->close();
        return $data;
    }

    public function getAllSubcategoriesWithCategoryNames() {
        $subCategories = [];
        $sql = "SELECT s.id, s.name, s.description, c.name as category_name, s.categoryId FROM subcategories s JOIN categories c ON s.categoryId = c.id ORDER BY s.categoryId, s.name";
        $result = $this->conn->query($sql);

        if ($result && $result->num_rows > 0) {
            while($row = $result->fetch_assoc()) {
                $subCategories[] = $row;
            }
        }
        return $subCategories;
    }

    public function getSubcategoriesByCategoryId($categoryId) {
        $subCategories = [];
        $stmt = $this->conn->prepare("SELECT id, categoryId, name, description FROM subcategories WHERE categoryId = ? ORDER BY name ASC");
        $stmt->bind_param("i", $categoryId);
        $stmt->execute();
        $result = $stmt->get_result();

        if ($result && $result->num_rows > 0) {
            while($row = $result->fetch_assoc()) {
                $subCategories[] = $row;
            }
        }
        $stmt->close();
        return $subCategories;
    }

    private function isSubcategoryNameTaken($categoryId, $name, $excludeId = null) {
        $sql = "SELECT id FROM subcategories WHERE categoryId = ? AND name = ?";
        $params = [$categoryId, $name];
        $types = "is";

        if ($excludeId !== null) {
            $sql .= " AND id != ?";
            $params[] = $excludeId;
            $types .= "i";
        }

        $stmt = $this->conn->prepare($sql);
        $stmt->bind_param($types, ...$params);
        $stmt->execute();
        $stmt->store_result();
        $is_taken = $stmt->num_rows > 0;
        $stmt->close();
        return $is_taken;
    }

    // Diğer kategori ile ilgili fonksiyonlar buraya eklenebilir (örn: getAllCategories)
    public function getAllCategories() {
        $categories = [];
        $sql_categories = "SELECT id, name FROM categories ORDER BY name";
        $result_categories = $this->conn->query($sql_categories);

        if ($result_categories && $result_categories->num_rows > 0) {
            while($row_cat = $result_categories->fetch_assoc()) {
                $categories[] = $row_cat;
            }
        }
        return $categories;
    }

    // Alt kategori adına göre ID getiren metot (CSV yükleme için gerekli)
    public function getSubcategoryByName($name) {
        $stmt = $this->conn->prepare("SELECT id, categoryId FROM subcategories WHERE name = ? LIMIT 1");
        $stmt->bind_param("s", $name);
        $stmt->execute();
        $result = $stmt->get_result();
        $data = $result->fetch_assoc();
        $stmt->close();
        return $data; // Found subcategory (with id and categoryId) or null
    }
}

?> 