<?php

require_once __DIR__ . '/../db_connect.php';
require_once __DIR__ . '/FcmService.php';

class QuestionAnswerService {
    private $conn;
    private $fcmService;

    public function __construct($db_connection) {
        $this->conn = $db_connection;
        $this->fcmService = new FcmService($db_connection);
    }

    // Question Answer CRUD
    public function addQuestionAnswer($subCategoryId, $packageId, $question, $answer) {
        // Soru-cevabın benzersizliğini kontrol et (aynı alt kategori ve paket içinde)
        $sql_check = "SELECT id FROM question_answers WHERE subCategoryId = ? AND question = ?";
        $check_params = [$subCategoryId, $question];
        $check_types = "is";

        if ($packageId !== NULL) {
            $sql_check .= " AND packageId = ?";
            $check_params[] = $packageId;
            $check_types .= "i";
        } else {
            $sql_check .= " AND packageId IS NULL";
        }

        $stmt_check = $this->conn->prepare($sql_check);
        $stmt_check->bind_param($check_types, ...$check_params);
        $stmt_check->execute();
        $stmt_check->store_result();
        if ($stmt_check->num_rows > 0) {
            $stmt_check->close();
            return ['status' => 'error', 'message' => "Bu soru, seçilen alt kategori ve paket içinde zaten mevcut."];
        } else {
            $stmt_check->close();
            $stmt = $this->conn->prepare("INSERT INTO question_answers (subCategoryId, packageId, question, answer) VALUES (?, ?, ?, ?)");
            $stmt->bind_param("iiss", $subCategoryId, $packageId, $question, $answer);

            if ($stmt->execute()) {
                $newId = $stmt->insert_id;
                $stmt->close();
                return ['status' => 'success', 'message' => "Soru-Cevap başarıyla eklendi!", 'id' => $newId];
            } else {
                $error = $stmt->error;
                $stmt->close();
                return ['status' => 'error', 'message' => "Soru-Cevap eklenirken hata: " . $error];
            }
        }
    }

    public function updateQuestionAnswer($id, $subCategoryId, $question, $answer, $packageId) {
        // Soru-cevabın benzersizliğini kontrol et (aynı alt kategori ve paket içinde, kendisi hariç)
        $sql_check = "SELECT id FROM question_answers WHERE subCategoryId = ? AND question = ? AND id != ?";
        $check_params = [$subCategoryId, $question, $id];
        $check_types = "isi";

        if ($packageId !== NULL) {
            $sql_check .= " AND packageId = ?";
            $check_params[] = $packageId;
            $check_types .= "i";
        } else {
            $sql_check .= " AND packageId IS NULL";
        }

        $stmt_check = $this->conn->prepare($sql_check);
        $stmt_check->bind_param($check_types, ...$check_params);
        $stmt_check->execute();
        $stmt_check->store_result();
        if ($stmt_check->num_rows > 0) {
            $stmt_check->close();
            return ['status' => 'error', 'message' => "Bu soru, seçilen alt kategori ve pakette zaten başka bir soru-cevap tarafından kullanılıyor."];
        } else {
            $stmt_check->close();
            $stmt = $this->conn->prepare("UPDATE question_answers SET subCategoryId = ?, packageId = ?, question = ?, answer = ? WHERE id = ?");
            $stmt->bind_param("iissi", $subCategoryId, $packageId, $question, $answer, $id);

            if ($stmt->execute()) {
                $affected_rows = $stmt->affected_rows;
                $stmt->close();
                if ($affected_rows > 0) {
                    return ['status' => 'success', 'message' => "Soru-Cevap başarıyla güncellendi!"];
                } else {
                    return ['status' => 'info', 'message' => "Güncellenecek soru-cevap bulunamadı veya veri değişmedi."];
                }
            } else {
                $error = $stmt->error;
                $stmt->close();
                return ['status' => 'error', 'message' => "Soru-Cevap güncellenirken hata: " . $error];
            }
        }
    }

    public function deleteQuestionAnswer($id) {
        $stmt = $this->conn->prepare("DELETE FROM question_answers WHERE id = ?");
        $stmt->bind_param("i", $id);
        if ($stmt->execute()) {
            $affected_rows = $stmt->affected_rows;
            $stmt->close();
            if ($affected_rows > 0) {
                return ['status' => 'success', 'message' => "Soru-Cevap başarıyla silindi!"];
            } else {
                return ['status' => 'error', 'message' => "Silinecek soru-cevap bulunamadı."];
            }
        } else {
            $error = $stmt->error;
            $stmt->close();
            return ['status' => 'error', 'message' => "Soru-Cevap silinirken hata: " . $error];
        }
    }

    // Toplu Soru-Cevap Silme
    public function deleteMultipleQuestionAnswers(array $ids) {
        if (empty($ids)) {
            return ['status' => 'warning', 'message' => "Silmek için hiçbir ID sağlanmadı."];
        }

        // ID listesi için placeholder oluştur
        $placeholders = implode(',', array_fill(0, count($ids), '?'));
        $types = str_repeat('i', count($ids));

        $sql = "DELETE FROM question_answers WHERE id IN ($placeholders)";
        $stmt = $this->conn->prepare($sql);
        $stmt->bind_param($types, ...$ids);

        if ($stmt->execute()) {
            $affected_rows = $stmt->affected_rows;
            $stmt->close();
            return ['status' => 'success', 'message' => "Başarıyla silindi: {$affected_rows} soru-cevap."];
        } else {
            $error = $stmt->error;
            $stmt->close();
            return ['status' => 'error', 'message' => "Toplu silme sırasında hata: " . $error];
        }
    }

    // Toplu Soru-Cevap Paketini Güncelleme
    public function updateMultipleQuestionAnswerPackage(array $ids, ?int $packageId) {
         if (empty($ids)) {
            return ['status' => 'warning', 'message' => "Paketini değiştirmek için hiçbir ID sağlanmadı."];
        }

        // ID listesi için placeholder oluştur
        $placeholders = implode(',', array_fill(0, count($ids), '?'));
        $types = str_repeat('i', count($ids));
        $params = $ids;

        $sql = "UPDATE question_answers SET packageId = ? WHERE id IN ($placeholders)";

        // packageId NULL ise tipi 'i' yerine 's' gibi davranması gerekebilir veya NULL için ayrı ele alınmalı.
        // mysqli bind_param NULL için özel bir durum gerektirir. 'i' tipi NULL değerleri kabul etse de...
        // Daha güvenli bir yaklaşım, packageId için tipi ayrı belirlemektir.
        
        $update_types = 'i' . $types; // packageId için 'i' (veya NULL için 's' gibi düşünelim) + idler için 'i' ler
        $update_params = [$packageId];
        $update_params = array_merge($update_params, $params);

        // mysqli bind_param doğrudan null değeri için 'i' tipini kabul eder, ekstra bir işleme gerek yok.

        $stmt = $this->conn->prepare($sql);
        $stmt->bind_param($update_types, ...$update_params);

        if ($stmt->execute()) {
            $affected_rows = $stmt->affected_rows;
            $stmt->close();
             if ($packageId === NULL) {
                 return ['status' => 'success', 'message' => "Başarıyla paketsiz yapıldı: {$affected_rows} soru-cevap."];
             } else {
                 return ['status' => 'success', 'message' => "Başarıyla paketi değiştirildi: {$affected_rows} soru-cevap."];
             }
        } else {
            $error = $stmt->error;
            $stmt->close();
            return ['status' => 'error', 'message' => "Toplu paket değiştirme sırasında hata: " . $error];
        }
    }

    public function getQuestionAnswerById($id) {
        $stmt = $this->conn->prepare("SELECT qa.*, s.name as subcategory_name FROM question_answers qa JOIN subcategories s ON qa.subCategoryId = s.id WHERE qa.id = ?");
        $stmt->bind_param("i", $id);
        $stmt->execute();
        $result = $stmt->get_result();
        $data = $result->fetch_assoc();
        $stmt->close();
        return $data;
    }

    public function getQuestionAnswersByFilters($subCategoryId = 0, $packageId = 0, $includeNullPackageId = false) {
        $sql = "SELECT qa.id, qa.question, qa.answer, qa.subCategoryId, qa.packageId, s.name as subcategory_name, qp.title as package_title FROM question_answers qa JOIN subcategories s ON qa.subCategoryId = s.id LEFT JOIN question_packages qp ON qa.packageId = qp.id";
        $params = [];
        $types = "";
        $whereClauses = [];

        if ($subCategoryId > 0) {
            $whereClauses[] = "qa.subCategoryId = ?";
            $params[] = $subCategoryId;
            $types .= "i";
        }

        if ($packageId > 0) {
            $whereClauses[] = "qa.packageId = ?";
            $params[] = $packageId;
            $types .= "i";
        } else if ($includeNullPackageId) { // packageId null olanları getir
            $whereClauses[] = "qa.packageId IS NULL";
        }

        if (count($whereClauses) > 0) {
            $sql .= " WHERE " . implode(" AND ", $whereClauses);
        }

        $sql .= " ORDER BY qa.subCategoryId, qa.packageId, qa.id";

        $stmt = $this->conn->prepare($sql);
        if ($params) {
            $stmt->bind_param($types, ...$params);
        }
        $stmt->execute();
        $result = $stmt->get_result();

        $questions = [];
        while ($row = $result->fetch_assoc()) {
            $questions[] = $row;
        }
        $stmt->close();
        return $questions;
    }

    // Raporlama Metotları

    // Kategori Bazında Soru Sayısı
    public function getQuestionCountByCategory() {
        $sql = "SELECT c.name as category_name, COUNT(qa.id) as question_count
                FROM categories c
                JOIN subcategories s ON c.id = s.categoryId
                LEFT JOIN question_answers qa ON s.id = qa.subCategoryId
                GROUP BY c.id, c.name
                ORDER BY c.name ASC";
        
        $result = $this->conn->query($sql);
        $reportData = [];
        if ($result && $result->num_rows > 0) {
            while ($row = $result->fetch_assoc()) {
                $reportData[] = $row;
            }
        }
        return $reportData;
    }

    // Alt Kategori Bazında Soru Sayısı
    public function getQuestionCountBySubcategory() {
         $sql = "SELECT s.name as subcategory_name, c.name as category_name, COUNT(qa.id) as question_count
                FROM subcategories s
                JOIN categories c ON s.categoryId = c.id
                LEFT JOIN question_answers qa ON s.id = qa.subCategoryId
                GROUP BY s.id, s.name, c.name
                ORDER BY c.name ASC, s.name ASC";

        $result = $this->conn->query($sql);
        $reportData = [];
        if ($result && $result->num_rows > 0) {
            while ($row = $result->fetch_assoc()) {
                $reportData[] = $row;
            }
        }
        return $reportData;
    }

    // Paket Bazında Soru Sayısı
    public function getQuestionCountByPackage() {
        $sql = "SELECT qp.title as package_title, s.name as subcategory_name, COUNT(qa.id) as question_count
                FROM question_packages qp
                JOIN subcategories s ON qp.subCategoryId = s.id
                LEFT JOIN question_answers qa ON qp.id = qa.packageId
                GROUP BY qp.id, qp.title, s.name
                ORDER BY s.name ASC, qp.title ASC";

        $result = $this->conn->query($sql);
        $reportData = [];
        if ($result && $result->num_rows > 0) {
            while ($row = $result->fetch_assoc()) {
                $reportData[] = $row;
            }
        }
        return $reportData;
    }

    // Son Eklenen Sorular
    public function getLatestQuestions(int $limit = 10) {
        $sql = "SELECT qa.id, qa.question, qa.answer, s.name as subcategory_name, qp.title as package_title
                FROM question_answers qa
                JOIN subcategories s ON qa.subCategoryId = s.id
                LEFT JOIN question_packages qp ON qa.packageId = qp.id
                ORDER BY qa.id DESC
                LIMIT ?";

        $stmt = $this->conn->prepare($sql);
        $stmt->bind_param("i", $limit);
        $stmt->execute();
        $result = $stmt->get_result();
        
        $questions = [];
        if ($result && $result->num_rows > 0) {
             while ($row = $result->fetch_assoc()) {
                // Cevap ve Soru metinlerini kısalt
                 $row['question'] = htmlspecialchars(mb_substr($row['question'], 0, 100)) . (mb_strlen($row['question']) > 100 ? '...' : '');
                 $row['answer'] = htmlspecialchars(mb_substr($row['answer'], 0, 100)) . (mb_strlen($row['answer']) > 100 ? '...' : '');
                $questions[] = $row;
            }
        }
        $stmt->close();
        return $questions;
    }

    // Question Package CRUD
    public function addQuestionPackage($subCategoryId, $title, $description) {
        // Paket başlığının benzersizliğini kontrol et (aynı alt kategori içinde)
        if ($this->isQuestionPackageTitleTaken($subCategoryId, $title)) {
            return ['status' => 'error', 'message' => "Bu alt kategoride aynı başlığa sahip bir paket zaten mevcut."];
        }

        $stmt = $this->conn->prepare("INSERT INTO question_packages (subCategoryId, title, description) VALUES (?, ?, ?)");
        $stmt->bind_param("iss", $subCategoryId, $title, $description);

        if ($stmt->execute()) {
            $newId = $stmt->insert_id;
            $stmt->close();

            // Yeni soru paketi eklendiğinde bildirim gönder
            $notificationTitle = "Yeni Soru Paketi Eklendi";
            $notificationBody = "Yeni soru paketi: \"$title\" eklendi! Hemen göz atın.";
            $this->fcmService->sendNotification($notificationTitle, $notificationBody, ['type' => 'new_package', 'packageId' => $newId]);

            return ['status' => 'success', 'message' => "Soru paketi başarıyla eklendi!", 'id' => $newId];
        } else {
            $error = $stmt->error;
            $stmt->close();
            return ['status' => 'error', 'message' => "Soru paketi eklenirken hata: " . $error];
        }
    }

    public function updateQuestionPackage($id, $subCategoryId, $title, $description) {
        // Paket başlığının benzersizliğini kontrol et (aynı alt kategori içinde, kendisi hariç)
        if ($this->isQuestionPackageTitleTaken($subCategoryId, $title, $id)) {
            return ['status' => 'error', 'message' => "Bu alt kategoride aynı başlığa sahip başka bir paket zaten mevcut."];
        }

        $stmt = $this->conn->prepare("UPDATE question_packages SET subCategoryId = ?, title = ?, description = ? WHERE id = ?");
        $stmt->bind_param("issi", $subCategoryId, $title, $description, $id);

        if ($stmt->execute()) {
            $affected_rows = $stmt->affected_rows;
            $stmt->close();
            if ($affected_rows > 0) {
                return ['status' => 'success', 'message' => "Soru paketi başarıyla güncellendi!"];
            } else {
                return ['status' => 'info', 'message' => "Güncellenecek soru paketi bulunamadı veya veri değişmedi."];
            }
        } else {
            $error = $stmt->error;
            $stmt->close();
            return ['status' => 'error', 'message' => "Soru paketi güncellenirken hata: " . $error];
        }
    }

    public function deleteQuestionPackage($packageId) {
        // Pakete bağlı soru-cevapları sil (CASCADE ile otomatik silinmeli, ancak kontrol için bırakılabilir)
        $stmt = $this->conn->prepare("DELETE FROM question_packages WHERE id = ?");
        $stmt->bind_param("i", $packageId);
        if ($stmt->execute()) {
            $affected_rows = $stmt->affected_rows;
            $stmt->close();
            if ($affected_rows > 0) {
                return ['status' => 'success', 'message' => "Soru paketi ve ilişkili soru-cevaplar başarıyla silindi!"];
            } else {
                return ['status' => 'error', 'message' => "Silinecek soru paketi bulunamadı."];
            }
        } else {
            $error = $stmt->error;
            $stmt->close();
            return ['status' => 'error', 'message' => "Soru paketi silinirken hata: " . $error];
        }
    }

    public function getQuestionPackageById($packageId) {
        $stmt = $this->conn->prepare("SELECT qp.*, s.name as subcategory_name FROM question_packages qp JOIN subcategories s ON qp.subCategoryId = s.id WHERE qp.id = ?");
        $stmt->bind_param("i", $packageId);
        $stmt->execute();
        $result = $stmt->get_result();
        $data = $result->fetch_assoc();
        $stmt->close();
        return $data;
    }

    public function getQuestionPackagesBySubcategoryId($subCategoryId) {
        $stmt = $this->conn->prepare("SELECT qp.*, s.name as subcategory_name FROM question_packages qp JOIN subcategories s ON qp.subCategoryId = s.id WHERE qp.subCategoryId = ? ORDER BY qp.title ASC");
        $stmt->bind_param("i", $subCategoryId);
        $stmt->execute();
        $result = $stmt->get_result();
        $packages = [];
        while ($row = $result->fetch_assoc()) {
            $packages[] = $row;
        }
        $stmt->close();
        return $packages;
    }

    public function getAllQuestionPackages() {
        $questionPackages = [];
        $sql_packages = "SELECT qp.id, qp.subCategoryId, qp.title, qp.description, s.name as subcategory_name
                         FROM question_packages qp
                         JOIN subcategories s ON qp.subCategoryId = s.id
                         ORDER BY s.name ASC, qp.title ASC";
        $result_packages = $this->conn->query($sql_packages);
        if ($result_packages && $result_packages->num_rows > 0) {
            while ($row = $result_packages->fetch_assoc()) {
                $questionPackages[] = $row;
            }
        }
        return $questionPackages;
    }

    public function getQuestionPackageByTitleAndSubcategoryId($title, $subCategoryId) {
        $stmt = $this->conn->prepare("SELECT id FROM question_packages WHERE title = ? AND subCategoryId = ?");
        $stmt->bind_param("si", $title, $subCategoryId);
        $stmt->execute();
        $result = $stmt->get_result();
        $data = $result->fetch_assoc();
        $stmt->close();
        return $data;
    }

    // Service fonksiyonları dışından çağrılmaması gereken yardımcı metotlar
    private function isQuestionPackageTitleTaken($subCategoryId, $title, $excludeId = null) {
        $sql = "SELECT id FROM question_packages WHERE subCategoryId = ? AND title = ?";
        $params = [$subCategoryId, $title];
        $types = "is";

        if ($excludeId !== null) {
            $sql .= " AND id != ?";
            $params[] = $excludeId;
            $types .= "i";
        }

        $stmt = $this->conn->prepare($sql);
        $stmt->bind_param($types, ...$params);
        $stmt->execute();
        $stmt->store_result();
        $isTaken = $stmt->num_rows > 0;
        $stmt->close();
        return $isTaken;
    }

    // Subcategory methods (likely belong in a separate SubcategoryService, but kept here for now)
    public function getAllSubcategories() {
        $subcategories = [];
        $sql = "SELECT * FROM subcategories ORDER BY name ASC";
        $result = $this->conn->query($sql);
        if ($result && $result->num_rows > 0) {
            while ($row = $result->fetch_assoc()) {
                $subcategories[] = $row;
            }
        }
        return $subcategories;
    }

    public function getAllSubcategoriesWithCategoryNames() {
        $subcategories = [];
        $sql = "SELECT s.*, c.name as category_name FROM subcategories s JOIN categories c ON s.categoryId = c.id ORDER BY c.name ASC, s.name ASC";
        $result = $this->conn->query($sql);
        if ($result && $result->num_rows > 0) {
            while ($row = $result->fetch_assoc()) {
                $subcategories[] = $row;
            }
        }
        return $subcategories;
    }

    public function getSubcategoryByName($name) {
        $stmt = $this->conn->prepare("SELECT id FROM subcategories WHERE name = ?");
        $stmt->bind_param("s", $name);
        $stmt->execute();
        $result = $stmt->get_result();
        $data = $result->fetch_assoc();
        $stmt->close();
        return $data;
    }

    // Category methods (likely belong in a separate CategoryService, but kept here for now)
    public function getAllCategories() {
        $categories = [];
        $sql = "SELECT * FROM categories ORDER BY name ASC";
        $result = $this->conn->query($sql);
        if ($result && $result->num_rows > 0) {
            while ($row = $result->fetch_assoc()) {
                $categories[] = $row;
            }
        }
        return $categories;
    }

    // Get total counts for dashboard (likely belong elsewhere)
    public function getTotalCount($table) {
        // Bu metot genel amaçlı değil, güvenlik açığı olabilir.
        // Belirli tablolar için özel metotlar yazmak daha iyidir.
        // Örnek: getTotalQuestionAnswersCount(), getTotalUsersCount()
        $allowed_tables = ['question_answers', 'question_packages', 'categories', 'subcategories', 'users'];
        if (!in_array($table, $allowed_tables)) {
            return 0; // veya hata döndür
        }
        $sql = "SELECT COUNT(*) as count FROM " . $table;
        $result = $this->conn->query($sql);
        if ($result && $result->num_rows > 0) {
            $row = $result->fetch_assoc();
            return $row['count'];
        }
        return 0;
    }

    public function getTotalQuestionAnswersCount() {
         $sql = "SELECT COUNT(*) as count FROM question_answers";
        $result = $this->conn->query($sql);
        if ($result && $result->num_rows > 0) {
            $row = $result->fetch_assoc();
            return $row['count'];
        }
        return 0;
    }

     public function getTotalQuestionPackagesCount() {
         $sql = "SELECT COUNT(*) as count FROM question_packages";
        $result = $this->conn->query($sql);
        if ($result && $result->num_rows > 0) {
            $row = $result->fetch_assoc();
            return $row['count'];
        }
        return 0;
    }

     public function getTotalCategoriesCount() {
         $sql = "SELECT COUNT(*) as count FROM categories";
        $result = $this->conn->query($sql);
        if ($result && $result->num_rows > 0) {
            $row = $result->fetch_assoc();
            return $row['count'];
        }
        return 0;
    }

    public function getTotalSubcategoriesCount() {
         $sql = "SELECT COUNT(*) as count FROM subcategories";
        $result = $this->conn->query($sql);
        if ($result && $result->num_rows > 0) {
            $row = $result->fetch_assoc();
            return $row['count'];
        }
        return 0;
    }

     public function getTotalUsersCount() {
         $sql = "SELECT COUNT(*) as count FROM users";
        $result = $this->conn->query($sql);
        if ($result && $result->num_rows > 0) {
            $row = $result->fetch_assoc();
            return $row['count'];
        }
        return 0;
    }
}

?> 